<?php

namespace App\Http\Controllers\Api\Report;

use App\Core\TenantCore;
use App\Helper\UtilHelper;
use App\Http\Controllers\Controller;
use App\Models\Tenant\Catalog\Product\ProductRepository;
use App\Models\Tenant\Catalog\ProductStock\ProductStockRepository;
use App\Models\Tenant\Config\TypeVoucher\TypeVoucherRepository;
use App\Models\Tenant\Money\Movement\MovementRepository;
use App\Models\Tenant\Purchase\Purchase\PurchaseRepository;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;

class AccountingReportController extends Controller
{
    protected $tenant;
    protected $utilHelper;
    protected $rMovement;
    protected $rPurchase;
    protected $rProductStock;
    protected $rProduct;
    protected $rTypeVoucher;

    public function __construct()
    {
        $this->tenant = new TenantCore();
        $this->utilHelper = new UtilHelper();
        $this->rMovement = new MovementRepository();
        $this->rPurchase = new PurchaseRepository();
        $this->rProductStock = new ProductStockRepository();
        $this->rProduct = new ProductRepository();
        $this->rTypeVoucher = new TypeVoucherRepository();
    }


    public function listVouchers(Request $request){
        $item = $this->rMovement->listVouchers($request);
        return response()->json($item);
    }

    public function listVouchersExport(Request $request){

        $company = $this->tenant->company();
        $format = $request->get('format');
        $parameters = $request->all();

        $cols = array(
            "date_movement" => array("col" => "A", "label" => "Fecha", "pos" => "C", "width" => 18)
        , "typevoucher" => array("col" => "B", "label" => "Tipo Comprobante", "pos" => "C", "width" => 35)
        , "serie" => array("col" => "C", "label" => "Serie", "pos" => "C", "width" => 10)
        , "nro_voucher" => array("col" => "C", "label" => "Voucher", "pos" => "C", "width" => 17)
        , "client_document" => array("col" => "D", "label" => "DNI/RUC", "pos" => "C", "width" => 25)
        , "client" => array("col" => "E", "label" => "Cliente", "pos" => "L", "width" => 61)
        , "status" => array("col" => "F", "label" => "Estado", "pos" => "C", "width" => 15)
        , "amount" => array("col" => "G", "label" => "Total", "pos" => "R", "width" => 20)
        );

        $title_document = "FACTURACION ELECTRONICA";

        $rows =  $this->rMovement->listVouchers($request, true);
        $rows =  $rows['results'];

        if ($format == config('app.export_format_excel')) {

            $style_bold = array("font"=>array("bold"=>true));

            $excel = new Spreadsheet();
            $excel->getActiveSheet()->setCellValue('A1', $title_document);
            $excel->getActiveSheet()->getStyle('A1')->applyFromArray($style_bold);
            $excel->getActiveSheet()->setCellValue("A2", 'Archivo Generado: '.date('Y-m-d h:i a'));


            $y = 4;
            foreach($cols as $val) {
                $this->utilHelper->row_excel($excel, $val["col"], $y, $val["label"], null, true, true);
            }

            if( ! empty($rows)) {
                $total = 0;

                foreach($rows as $row) {
                    $y ++;
                    foreach($cols as $key=>$val) {
                        $this->utilHelper->row_excel($excel, $val["col"], $y, $row->{$key}, null, false, true);
                    }

                    $total = $total + $row->amount;
                }
                $y++;

                $this->utilHelper->row_excel($excel, $cols["status"]["col"], $y, "TOTAL", null, true);
                $this->utilHelper->row_excel($excel, $cols["amount"]["col"], $y, $total, null, true);

            }

            $filename='facturacion_electronica-'.date("dmY-hi").'.xlsx';
            header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
            header('Content-Disposition: attachment;filename="'.$filename.'"');
            header('Cache-Control: max-age=0');
            $objWriter = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($excel);
            $objWriter->save('php://output');
            exit;
        }
        else{

            $pdf = new \App\Libraries\Pdf();
            $pdf->AliasNbPages(); // para el conteo de paginas
            $pdf->SetLeftMargin(4);
            $pdf->SetTopMargin(7);
            $pdf->SetTitle("FACTURACION ELECTRONICA", 12);
            $pdf->SetDrawColor(160, 160, 160);

            // espacios iniciales
            $ancho_total= 210;
            $height_total = 297;
            $espac = 5;
            $pos_initial_x = 5;
            $h_logo = 20;
            $w_logo=40;

            $pdf->AddPage();


            // fecha impresión
            $pdf->SetFont('Arial','',9);
            $pdf->SetY(10);
            $pdf->SetX($ancho_total-45);
            $pdf->MultiCell(50, 2.5, utf8_decode('Generado el '.date('Y-m-d')), 0);
            $pdf->SetY(15);
            $pdf->SetX($ancho_total-40);
            $pdf->MultiCell(50, 2.5, utf8_decode('Hora: '.date('H:i:s a ')), 0);

            $logo_empresa = 'default.jpg';

            if ($logo_empresa!=null || $logo_empresa!=''){
                $logo_empresa = "companies/".$company->url_image;
            }

            $pdf->Image($logo_empresa, $pos_initial_x, $espac, $w_logo, $h_logo);

            $pdf->SetY(30);

            if( ! empty($rows)) {
                $pdf->SetFillColor(246, 212, 212);

                $fields = array_keys($cols);
                $pos = array_column($cols, "pos");
                $total = 0;

                $pdf->SetFont('Arial','B',9);
                $pdf->SetWidths(array(22, 100));
                $pdf->Row(array('EMPRESA :', $company->social_reason), array("L"), "N","Y");

                $pdf->SetWidths(array(22, 100));
                $pdf->Row(array( utf8_decode('DIRECCIÓN :'), utf8_decode($company->address)), array("L"), "N","Y");

                $pdf->SetY(30);
                $pdf->SetX(130);
                $pdf->SetWidths(array(12, 100));
                $pdf->Row(array('RUC :', $company->ruc), array("L"), "N", "Y");
                $pdf->Row(array(''), array("L"), "N", "Y");

                $pdf->SetY(42);

                $pdf->SetFont('Arial','B',9);
                $pdf->SetFillColor(245, 245, 245);
                $pdf->SetWidths(array_column($cols, 'width'));
                $pdf->SetHeight(7);
                $pdf->Row(array_column($cols, 'label'), array_fill(0, count($cols), "C"), "Y", "Y", array(245, 245, 245));

                $pdf->SetHeight(6);
                $pdf->SetFillColor(255, 255, 255);
                foreach($rows as $row) {
                    $pdf->SetFont('Arial','',7);
                    $values = array();
                    $bg_row = array();
                    foreach($fields as $field) {
                        $values[] = isset($row->{$field}) ? utf8_decode($row->{$field}) : "";
                    }

                    if($row->status == 'Anulado'){
//                        $pdf->SetDrawColor(160, 160, 160);
                        $bg_row = array(246, 222, 222);
                    }

                    $pdf->SetFillColor(246, 222, 222);

                    $total = $total + $row->amount;
                    $pdf->Row($values, $pos, "Y", "Y", $bg_row);
                }

                $pdf->SetFont("Arial", "B", 8);
                $pdf->SetWidths(array(181, 20));
                $pdf->Row(array("TOTAL", number_format($total,2,".","'")), array("R", "R"));
            }

            $pdf->Output();
            exit;

        }

    }


    public function merchadiseValuation(Request $request){
        $movement_average_product = $this->rMovement->averagePriceProductWarehouseDetail($request);
        $purchase_average_product = $this->rPurchase->averagePriceProductPurchaseDetail($request);
        $stock_products = $this->rProductStock->listStockAvailablesFromProduct();
        $response = array(
            'status' => 'success',
            'movement_average_product' => $movement_average_product,
            'purchase_average_product' => $purchase_average_product,
            'stock_products' => $stock_products,
        );
        return response()->json($response);
    }


    public function saleByVouchers(Request $request){
        $sale_by_voucher = $this->rTypeVoucher->saleByVoucher($request);

        $response = array(
            'status' => 'success',
            'results' => $sale_by_voucher
        );
        return response()->json($response);
    }

    public function saleByVouchersExport(Request $request){

        $company = $this->tenant->company();
        $format = $request->get('format');
        $parameters = $request->all();

        $cols = array(
            "description" => array("col" => "A", "label" => "COMPROBANTE", "pos" => "C", "width" => 50)
            ,"quantity_sale" => array("col" => "B", "label" =>"NRO TRANSACCIONES", "pos" => "C", "width" => 40)
            ,"total_sale" => array("col" => "C", "label" => "TOTAL RECAUDADO", "pos" => "C", "width" => 40)
        );
        $rows = $this->rTypeVoucher->saleByVoucher($request);

        if ($format == config('app.export_format_excel')) {
            $subtitle = "";
            if(trim($request->start_range) != "")
                $subtitle .= "Desde ".$request->start_range." ";
            if(trim($request->end_range) != "")
                $subtitle .= "Al ".$request->end_range;

            $style_bold = array("font"=>array("bold"=>true));

            $excel = new Spreadsheet();
            $excel->getActiveSheet()->setCellValue('A1', "RECAUDACIÓN POR COMPROBANTES EMITIDOS {$subtitle}");
            $excel->getActiveSheet()->getStyle('A1')->applyFromArray($style_bold);
            $excel->getActiveSheet()->setCellValue("A2", 'Archivo Generado: '.date('Y-m-d h:i a'));


            $y = 4;
            foreach($cols as $val) {
                $this->utilHelper->row_excel($excel, $val["col"], $y, $val["label"], null, true, true);
            }

            if( ! empty($rows)) {
                $total = 0;
                foreach($rows as $row) {
                    $y ++;
                    foreach($cols as $key=>$val) {
                        $this->utilHelper->row_excel($excel, $val["col"], $y, $row->{$key}, null, false, true);
                    }
                    $total = $total + $row->total_sale;
                }
                $y++;

                $this->utilHelper->row_excel($excel, $cols["quantity_sale"]["col"], $y, "TOTAL", null, true);
                $this->utilHelper->row_excel($excel, $cols["total_sale"]["col"], $y, $total, null, true);
            }

            $filename='recaudacion_comprobantes-'.date("dmYhi").'.xlsx';
            header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
            header('Content-Disposition: attachment;filename="'.$filename.'"');
            header('Cache-Control: max-age=0');
            $objWriter = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($excel);
            $objWriter->save('php://output');
            exit;
        }
        else{

            $pdf = new \App\Libraries\Pdf();
            $pdf->AliasNbPages(); // para el conteo de paginas
            $pdf->SetLeftMargin(4);
            $pdf->SetTopMargin(7);
            $pdf->SetTitle(utf8_decode("RECAUDACION POR COMPROBANTES EMITIDOS"), 12);
            $pdf->SetDrawColor(160, 160, 160);

            // espacios iniciales
            $ancho_total= 210;
            $height_total = 297;
            $espac = 5;
            $pos_initial_x = 5;
            $h_logo = 20;
            $w_logo=40;

            $pdf->AddPage();

            $pdf->SetFont('Arial','',11);
            $pdf->SetY(13);
            $pdf->SetWidths(array(195));
            $pdf->Row(array('Desde el '.date("d M Y", strtotime($parameters['start_range'])).'   al  '.date("d M Y", strtotime($parameters['end_range']))), array("C"), "N","Y");

            // fecha impresión
            $pdf->SetFont('Arial','',9);
            $pdf->SetY(10);
            $pdf->SetX($ancho_total-45);
            $pdf->MultiCell(50, 2.5, utf8_decode('Generado el '.date('Y-m-d')), 0);
            $pdf->SetY(15);
            $pdf->SetX($ancho_total-40);
            $pdf->MultiCell(50, 2.5, utf8_decode('Hora: '.date('H:i:s a ')), 0);

            $logo_empresa = 'default.jpg';

            if ($logo_empresa!=null || $logo_empresa!=''){
                $logo_empresa = "companies/".$company->url_image;
            }

            $pdf->Image($logo_empresa, $pos_initial_x, $espac, $w_logo, $h_logo);

            $pdf->SetY(30);

            if( ! empty($rows)) {
                $pdf->SetFillColor(246, 212, 212);

                $fields = array_keys($cols);
                $pos = array_column($cols, "pos");
                $total = 0;

                $pdf->SetFont('Arial','B',9);
                $pdf->SetWidths(array(22, 100));
                $pdf->Row(array('EMPRESA :', $company->social_reason), array("L"), "N","Y");

                $pdf->SetWidths(array(22, 100));
                $pdf->Row(array( utf8_decode('DIRECCIÓN :'), utf8_decode($company->address)), array("L"), "N","Y");

                $pdf->SetY(30);
                $pdf->SetX(130);
                $pdf->SetWidths(array(12, 100));
                $pdf->Row(array('RUC :', $company->ruc), array("L"), "N", "Y");
                $pdf->Row(array(''), array("L"), "N", "Y");

                $pdf->SetY(42);

                $pdf->SetFont('Arial','B',9);
                $pdf->SetFillColor(245, 245, 245);
                $pdf->SetWidths(array_column($cols, 'width'));
                $pdf->SetHeight(7);
                $pdf->Row(array_column($cols, 'label'), array_fill(0, count($cols), "C"), "Y", "Y", array(245, 245, 245));

                $pdf->SetHeight(6);
                $pdf->SetFillColor(255, 255, 255);
                foreach($rows as $row) {
                    $pdf->SetFont('Arial','',7);
                    $values = array();
                    $bg_row = array();
                    foreach($fields as $field) {
                        $values[] = isset($row->{$field}) ? utf8_decode($row->{$field}) : "";
                    }

                    $pdf->SetFillColor(246, 222, 222);

                    $total = $total + $row->total_sale;
                    $pdf->Row($values, $pos, "Y", "Y", $bg_row);
                }

                $pdf->SetFont("Arial", "B", 10);
                $pdf->SetWidths(array(175, 20));
                $pdf->Row(array("TOTAL", number_format($total,2,".","'")), array("R", "R"),"N");
            }

            $pdf->Output();
            exit;

        }

    }

    public function valuedKardex(Request $request){

        $company = $this->tenant->company();
        set_time_limit(0);
        $pdf = new \App\Libraries\Pdf();

        $pdf->SetTitle(utf8_decode("KARDEX VALORIZADO"), 14, null, true);
        $pdf->AliasNbPages();

        $pdf->setFillColor(249, 249, 249);
        $pdf->SetDrawColor(204, 204, 204);

        $pdf->AddPage("L");
        $pdf->SetFillColor(255, 255, 255);

        $pos_initial_x = 3;
        $espac = 4;
        $h_logo = 20;
        $w_logo = 40;

        $logo_empresa = 'default.jpg';

        if ($logo_empresa != null || $logo_empresa != '') {
            $logo_empresa = "companies/" . $company->url_image;
        }

        $pdf->Image($logo_empresa, $pos_initial_x, $espac, $w_logo, $h_logo);


        // fecha impresión
        $pdf->SetFont('Arial', '', 9);
        $pdf->SetY(10);
        $pdf->SetX(255);
        $pdf->MultiCell(50, 2.5, utf8_decode('Generado el ' . date('Y-m-d')), 0);
        $pdf->SetY(15);
        $pdf->SetX(260);
        $pdf->MultiCell(50, 2.5, utf8_decode('Hora: ' . date('H:i:s a ')), 0);

        $pdf->SetFont('Arial', 'B', 9);
        $pdf->SetY(17);
        $pdf->SetWidths(array(280));
        $pdf->Row(array($company->social_reason.' - RUC: '.$company->ruc), array("C"), "N", "Y");

//        $pdf->Row(array('Desde el ' . date("d M Y", strtotime($parameters['start_range'])) . '   al  ' . date("d M Y", strtotime($parameters['end_range']))), array("C"), "N", "Y");

        $cols_detail = $this->head_value_kardex();
        $cols_pos = array_column($cols_detail, "pos");

        $pdf->SetFont('Arial', 'B', 8);
        $pdf->SetFillColor(245, 245, 245);
        $pdf->SetY(30);
        $pdf->SetX($pos_initial_x);
        $pdf->SetHeight(7);
        $pdf->SetWidths(array_column($cols_detail, 'width'));
        $pdf->Row(array_column($cols_detail, 'label'), $cols_pos, "Y", "Y", array(245, 245, 245));

        $pdf->SetY(40);
        $pdf->SetHeight(5);
        $pdf->SetFont('Arial', '', 7);
        $pdf->SetFillColor(255, 255, 255);

        //products control stock
        $products_control = $this->rProduct->getProductsControlStock();
        $total_balance = 0;

        foreach ($products_control as $product) {
            $product_concat = $product->description." - ".$product->code;
            $pdf->SetFont('Arial', 'B', 8);
            $pdf->SetX($pos_initial_x);
            $pdf->SetWidths(array(290));
            $pdf->SetHeight(7);
            $pdf->Row(array(utf8_decode($product_concat)), array("L"), "N", "Y");


            $rows = $this->query_value_kardex($product->id, $request);
            $pdf->SetFont('Arial', '', 7);
            $diff_quantity = 0;
            $diff_total = 0;
            $sum_quantity_entry = 0;
            $sum_quantity_departure = 0;
            $sum_total_entry = 0;
            $sum_total_departure = 0;

            $pdf->SetHeight(5);
            foreach ($rows as $row) {
                $pdf->SetX($pos_initial_x);

                $row->quantity_entry = null;
                $row->price_entry = null;
                $row->total_entry = null;

                $row->quantity_departure = null;
                $row->price_departure = null;
                $row->total_departure = null;

                if (trim($row->type_movement) == "E") {
                    $row->typemovement = 'INGRESO';
                    $row->quantity_entry = $row->quantity;
                    $row->price_entry = $row->price;
                    $total_entry = $row->price * $row->quantity;
                    $row->total_entry = $total_entry;

                    $diff_quantity = $diff_quantity + $row->quantity;
                    $diff_total = $diff_total + $total_entry;

                    $sum_quantity_entry = $sum_quantity_entry + $row->quantity;
                    $sum_total_entry = $sum_total_entry + $total_entry;

                } else if (trim($row->type_movement) == "S") {
                    $row->typemovement = 'SALIDA';
                    $row->quantity_departure = $row->quantity;
                    $row->price_departure = $row->price;
                    $total_departure = $row->price * $row->quantity;
                    $row->total_departure = $total_departure;

                    $diff_quantity = $diff_quantity - $row->quantity;
                    $diff_total = $diff_total - $total_departure;

                    $sum_quantity_departure = $sum_quantity_departure + $row->quantity;
                    $sum_total_departure = $sum_total_departure + $total_departure;
                }

                $item = array('date_'=>array( "label" => $row->date_)
                    ,'nro_voucher'=>array( "label" => $row->nro_voucher)
                    ,'typemovement'=>array( "label" => $row->typemovement)
                    ,'concept'=>array( "label" => utf8_decode($row->concept))
                    ,'quantity_entry'=>array( "label" => $row->quantity_entry)
                    ,'price_entry'=>array( "label" =>$row->price_entry)
                    ,'total_entry'=>array( "label" => $row->total_entry)
                    ,'quantity_departure'=>array( "label" => $row->quantity_departure)
                    ,'price_departure'=>array( "label" =>$row->price_departure)
                    ,'total_departure'=>array( "label" =>$row->total_departure)
                    ,'diff_quantity'=>array("label" => $diff_quantity)
                    ,'diff_total'=>array("label" => $diff_total)
                    );

                $pdf->SetWidths(array_column($cols_detail, 'width'));
                $pdf->Row(array_column($item, 'label'), $cols_pos, "Y", "Y");

            }

            // TOTALESS ================
            $pdf->SetFont('Arial', 'B', 7);
            $pdf->SetX(70+3);
            $pdf->SetHeight(6);
            $pdf->SetWidths(array(50, 20, 20, 20, 20, 20, 20, 25, 25));
            $pdf->Row(array("TOTALES",
                number_format($sum_quantity_entry, 2),
                "",
                'S/ '.number_format($sum_total_entry, 2),
                number_format($sum_quantity_departure, 2),
                "",
                'S/ '.number_format($sum_total_departure, 2),
                number_format($diff_quantity, 2),
                'S/ '.number_format($diff_total, 2),
                ),
                array("C", "C", "R", "R","C","C","R","C","R"),
                "Y", "Y");

            $total_balance = $total_balance + $diff_total;

            $pdf->SetFont('Arial', '', 7);
            $pdf->SetWidths(array(290));
            $pdf->Row(array(utf8_decode("")),array("L"), "N", "Y");

        }

        $pdf->SetWidths(array(290));
        $pdf->Row(array(utf8_decode("")),array("R"), "N", "Y");

        $pdf->SetFont('Arial', 'B', 10);
        $pdf->SetWidths(array(280));
        $pdf->Row(array(utf8_decode("VALORIZACIÓN TOTAL KARDEX : ".' S/'.number_format($total_balance,2))),array("R"), "N", "Y");


        $pdf->Output();
        exit;
    }

    public function head_value_kardex() {
        return array('date_'=>array("col" => "A", "label" => "FECHA", "pos" => "L", "width" => 20)
        ,'nro_voucher'=>array("col" => "A", "label" => utf8_decode("N° VOUCHER"), "pos" => "L", "width" => 25)
        ,'typemovement'=>array("col" => "A", "label" => "MOVIMIENTO", "pos" => "L", "width" => 25)
        ,'concept'=>array("col" => "A", "label" => "CONCEPTO", "pos" => "L", "width" => 50)
        ,'quantity_entry'=>array("col" => "A", "label" => "ENT. CANT", "pos" => "C", "width" => 20)
        ,'price_entry'=>array("col" => "A", "label" => "ENT. PU", "pos" => "R", "width" => 20)
        ,'total_entry'=>array("col" => "A", "label" => "ENT. PT", "pos" => "R", "width" => 20)
        ,'quantity_departure'=>array("col" => "A", "label" => "SAL. CANT", "pos" => "C", "width" => 20)
        ,'price_departure'=>array("col" => "A", "label" => "SAL. PU", "pos" => "R", "width" => 20)
        ,'total_departure'=>array("col" => "A", "label" => "SAL. PT", "pos" => "R", "width" => 20)
        ,'diff_quantity'=>array("col" => "A", "label" => "SALDO CANT", "pos" => "C", "width" => 25)
        ,'diff_total'=>array("col" => "A", "label" => "SALDO PT", "pos" => "R", "width" => 25)
        );
    }

    public function query_value_kardex($product_id , $parameters){


        $query = DB::connection('tenant')
            ->select("SELECT
                                warehouse_movements.date_movement AS date_,
                                warehouse_movements.type_movement AS type_movement,
                                warehouse_movements.nro_voucher AS nro_voucher,
                                warehouse_concepts.description AS concept,
                                warehouse_detailmovements.quantity AS quantity,
                                warehouse_detailmovements.price AS price,
                                (warehouse_detailmovements.price * warehouse_detailmovements.quantity) AS total_movement
                            FROM
                                warehouse_detailmovements
                            INNER JOIN catalog_products ON warehouse_detailmovements.product_id = catalog_products. ID
                            INNER JOIN warehouse_movements ON warehouse_detailmovements.warehousemovement_id = warehouse_movements. ID
                            INNER JOIN warehouse_concepts ON warehouse_movements.warehouseconcept_id = warehouse_concepts.id
                            WHERE catalog_products.id = ".$product_id." AND catalog_products.deleted_at ISNULL

                            UNION

                            SELECT
                                DATE(sale_sales.date_sale) AS date_,
                                concat('S') AS type_movement,
                                sale_sales.code AS nro_voucher,
                                concat('Venta de producto') AS concept,
                                sale_detailsales.quantity AS quantity,
                                sale_detailsales.price AS price,
                                (sale_detailsales.price * sale_detailsales.quantity) AS total_movement

                            FROM sale_detailsales
                            INNER JOIN catalog_products ON sale_detailsales.product_id = catalog_products.ID
                            INNER JOIN sale_sales ON sale_detailsales.sale_id = sale_sales.id
                            WHERE catalog_products.id = ".$product_id." AND catalog_products.deleted_at ISNULL

                            UNION

                            SELECT
                                DATE(purchase_purchases.date_purchase) AS date_,
                                concat('E') AS type_movement,
                                purchase_purchases.code AS nro_voucher,
                                concat('Compra de producto') AS concept,
                                purchase_detailpurchases.quantity AS quantity,
                                purchase_detailpurchases.price AS price,
                                (purchase_detailpurchases.price * purchase_detailpurchases.quantity) AS total_movement

                            FROM purchase_detailpurchases
                            INNER JOIN catalog_products ON purchase_detailpurchases.product_id = catalog_products.ID
                            INNER JOIN purchase_purchases ON purchase_detailpurchases.purchase_id = purchase_purchases.id
                            WHERE catalog_products.id = ".$product_id." AND catalog_products.deleted_at ISNULL

                            ORDER BY date_ ASC
                            ");

        return $query;
    }

    public function listVouchersDetail(Request $request){
        $item = $this->rMovement->listVouchersDetail($request);
        return response()->json($item);
    }

    public function listVouchersDetailExport(Request $request){

        $company = $this->tenant->company();
        $format = $request->get('format');
        $parameters = $request->all();

        $cols = array(
            "date_movement" => array("col" => "A", "label" => "Fecha", "pos" => "C", "width" => 18)
        , "nro_voucher" => array("col" => "B", "label" => "Voucher", "pos" => "C", "width" => 20)
        , "client" => array("col" => "C", "label" => "Nombre/Razon Social", "pos" => "L", "width" => 55)
        , "client_document" => array("col" => "D", "label" => "DNI/RUC", "pos" => "C", "width" => 25)
        , "total_gravado" => array("col" => "E", "label" => "T. Gravado", "pos" => "R", "width" => 20)
        , "total_igv" => array("col" => "F", "label" => "T. IGV", "pos" => "R", "width" => 20)
        , "total_exonerado" => array("col" => "G", "label" => "T.Exonerado", "pos" => "R", "width" => 22)
        , "total_sale" => array("col" => "H", "label" => "T. Venta", "pos" => "R", "width" => 22)
//        , "status" => array("col" => "H", "label" => "T. Venta", "pos" => "R", "width" => 22)
        );

        $cols_excel = array(
            "date_movement" => array("col" => "A", "label" => "Fecha", "pos" => "C", "width" => 18)
        , "nro_voucher" => array("col" => "B", "label" => "Voucher", "pos" => "C", "width" => 20)
        , "client" => array("col" => "C", "label" => "Nombre/Razon Social", "pos" => "L", "width" => 150)
        , "type_document" => array("col" => "D", "label" => "Tipo Doc.", "pos" => "L", "width" => 20)
        , "nro_document" => array("col" => "E", "label" => "Nro Documento", "pos" => "L", "width" => 30)
        , "status" => array("col" => "F", "label" => "Estado", "pos" => "L", "width" => 30)
        , "currency" => array("col" => "G", "label" => "Moneda", "pos" => "L", "width" => 20)
        , "no_affect_igv" => array("col" => "H", "label" => "No afecta IGV", "pos" => "R", "width" => 20)
        , "affect_igv" => array("col" => "I", "label" => "Afecta IGV", "pos" => "R", "width" => 20)
        , "total_gravado" => array("col" => "J", "label" => "T. Gravado", "pos" => "R", "width" => 20)
        , "total_igv" => array("col" => "K", "label" => "T. IGV", "pos" => "R", "width" => 20)
        , "total_exonerado" => array("col" => "L", "label" => "T.Exonerado", "pos" => "R", "width" => 22)
        , "total_sale" => array("col" => "M", "label" => "T. Venta", "pos" => "R", "width" => 22)
//        , "status" => array("col" => "H", "label" => "T. Venta", "pos" => "R", "width" => 22)
        );

        $title_document = "FACTURACION ELECTRONICA DETALLADA";

        $rows =  $this->rMovement->listVouchersDetail($request, true);
        $rows =  $rows['results'];

        $total_no_affect_igv = 0;
        $total_affect_igv = 0;
        $total_gravado = 0;
        $total_igv = 0;
        $total_exonerado = 0;
        $total_sale = 0;

        if ($format == config('app.export_format_excel')) {

            $style_bold = array("font"=>array("bold"=>true));

            $excel = new Spreadsheet();
            $excel->getActiveSheet()->setCellValue('A1', $title_document);
            $excel->getActiveSheet()->getStyle('A1')->applyFromArray($style_bold);
            $excel->getActiveSheet()->setCellValue("A2", 'Archivo Generado: '.date('Y-m-d h:i a'));


            $y = 4;
            foreach($cols_excel as $val) {
                $this->utilHelper->row_excel($excel, $val["col"], $y, $val["label"], null, true, true);
            }

            if( ! empty($rows)) {

                foreach($rows as $row) {
                    $y ++;

                    $row['total_gravado'] = $row->no_affect_igv + $row->affect_igv;

                    $row->status = "CONCRETADO";
                    $is_repayment = false;

                    if($row->date_repayment != null ){
                        $row->no_affect_igv = 0;
                        $row->affect_igv = 0;
                        $row->total_gravado = 0;
                        $row->total_igv = 0;
                        $row->total_sale = 0;

                        // set status
                        $row->status = "ANULADO";
                        $is_repayment = true;
                    }

//                    $row['total_sale'] = $total_no_affect_igv + $row->total_affect_igv + $row->total_igv;

                    foreach($cols_excel as $key=>$val) {
                        $this->utilHelper->row_excel($excel, $val["col"], $y, $row->{$key}, null, false, true, $is_repayment);
                    }

                    $total_no_affect_igv = $total_no_affect_igv + $row->no_affect_igv;
                    $total_affect_igv = $total_affect_igv + $row->affect_igv;
                    $total_gravado = $total_gravado + $row->total_gravado;
                    $total_igv = $total_igv + $row->total_igv;
                    $total_exonerado= $total_exonerado + $row->total_exonerado;
                    $total_sale= $total_sale + $row->total_sale;
                }

                $y++;

//                $this->utilHelper->row_excel($excel, $cols_excel["status"]["col"], $y, "TOTAL", null, true);
                $this->utilHelper->row_excel($excel, $cols_excel["no_affect_igv"]["col"], $y, $total_no_affect_igv, null, true);
                $this->utilHelper->row_excel($excel, $cols_excel["affect_igv"]["col"], $y, $total_affect_igv, null, true);
                $this->utilHelper->row_excel($excel, $cols_excel["total_gravado"]["col"], $y, $total_gravado, null, true);
                $this->utilHelper->row_excel($excel, $cols_excel["total_igv"]["col"], $y, $total_igv, null, true);
                $this->utilHelper->row_excel($excel, $cols_excel["total_exonerado"]["col"], $y, $total_exonerado, null, true);
                $this->utilHelper->row_excel($excel, $cols_excel["total_sale"]["col"], $y, $total_sale, null, true);
            }

//            $excel->getActiveSheet()->getStyle('A1:A5')->getFill()->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)->getStartColor()->setARGB('9A1A1A');


            $filename='facturacion_electronica_detallada-'.date("dmY-hi").'.xlsx';
            header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
            header('Content-Disposition: attachment;filename="'.$filename.'"');
            header('Cache-Control: max-age=0');
            $objWriter = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($excel);
            $objWriter->save('php://output');
            exit;
        }
        else{

            $pdf = new \App\Libraries\Pdf();
            $pdf->AliasNbPages(); // para el conteo de paginas
            $pdf->SetLeftMargin(4);
            $pdf->SetTopMargin(7);
            $pdf->SetTitle("FACTURACION ELECTRONICA DETALLADA", 12);
            $pdf->SetDrawColor(160, 160, 160);

            // espacios iniciales
            $ancho_total= 210;
            $height_total = 297;
            $espac = 5;
            $pos_initial_x = 5;
            $h_logo = 20;
            $w_logo=40;

            $pdf->AddPage();

            $pdf->SetFont('Arial', '', 11);
            $pdf->SetY(13);
            $pdf->SetWidths(array(195));
            $pdf->Row(array(utf8_decode('Periodo : '.$parameters['start_range'].' - '.$parameters['end_range'])), array("C"), "N", "Y");


            // fecha impresión
            $pdf->SetFont('Arial','',9);
            $pdf->SetY(10);
            $pdf->SetX($ancho_total-45);
            $pdf->MultiCell(50, 2.5, utf8_decode('Generado el '.date('Y-m-d')), 0);
            $pdf->SetY(15);
            $pdf->SetX($ancho_total-40);
            $pdf->MultiCell(50, 2.5, utf8_decode('Hora: '.date('H:i:s a ')), 0);

            $logo_empresa = 'default.jpg';

            if ($logo_empresa!=null || $logo_empresa!=''){
                $logo_empresa = "companies/".$company->url_image;
            }

            $pdf->Image($logo_empresa, $pos_initial_x, $espac, $w_logo, $h_logo);

            $pdf->SetY(30);

            if( ! empty($rows)) {
                $pdf->SetFillColor(246, 212, 212);

                $fields = array_keys($cols);
                $pos = array_column($cols, "pos");

                $pdf->SetFont('Arial','B',9);
                $pdf->SetWidths(array(22, 100));
                $pdf->Row(array('EMPRESA :', $company->social_reason), array("L"), "N","Y");

                $pdf->SetWidths(array(22, 100));
                $pdf->Row(array( utf8_decode('DIRECCIÓN :'), utf8_decode($company->address)), array("L"), "N","Y");

                $pdf->SetY(30);
                $pdf->SetX(130);
                $pdf->SetWidths(array(12, 100));
                $pdf->Row(array('RUC :', $company->ruc), array("L"), "N", "Y");
                $pdf->Row(array(''), array("L"), "N", "Y");

                $pdf->SetY(42);

                $pdf->SetFont('Arial','B',9);
                $pdf->SetFillColor(245, 245, 245);
                $pdf->SetWidths(array_column($cols, 'width'));
                $pdf->SetHeight(7);
                $pdf->Row(array_column($cols, 'label'), array_fill(0, count($cols), "C"), "Y", "Y", array(245, 245, 245));

                $pdf->SetHeight(6);
                $pdf->SetFillColor(255, 255, 255);
                foreach($rows as $row) {
                    $pdf->SetFont('Arial','',7);
                    $values = array();
                    $bg_row = array();

                    foreach($fields as $field) {
                        $values[] = isset($row->{$field}) ? utf8_decode($row->{$field}) : "";
                    }

                    if($row->status == 'Anulado'){
//                        $pdf->SetDrawColor(160, 160, 160);
                        $bg_row = array(246, 222, 222);
                    }

                    $pdf->SetFillColor(246, 222, 222);

                    $total_gravado = $total_gravado + $row->total_gravado;
                    $total_igv = $total_igv + $row->total_igv;
                    $total_exonerado= $total_exonerado + $row->total_exonerado;
                    $total_sale= $total_sale + $row->total_sale;

                    $pdf->Row($values, $pos, "Y", "Y", $bg_row);
                }

                $pdf->SetFont("Arial", "B", 8);
                $pdf->SetWidths(array(118,20,20, 22, 22));
                $pdf->Row(array(
                    "",
                    number_format($total_gravado,2,".","'"),
                    number_format($total_igv,2,".","'"),
                    number_format($total_exonerado,2,".","'"),
                    number_format($total_sale,2,".","'")), array("R", "R", "R", "R", "R"));
            }

            $pdf->Output();
            exit;

        }

    }
}
