<?php

namespace App\Models\Tenant\Sale\Client;

use App\Helper\HelperChecker;
use App\Helper\StorageHelper;
use App\Models\Contract\BaseRepositoryInterface;
use Illuminate\Support\Facades\DB;

class ClientRepository implements BaseRepositoryInterface
{

    protected $obj;
    protected $helperChecker;
    protected $storageHelper;

    public function __construct()
    {
        $this->obj = new Client();
        $this->helperChecker = new HelperChecker();
        $this->storageHelper = new StorageHelper();
    }

    public function model()
    {
        return $this->obj;
    }

    public function all()
    {
        return $this->obj->all();
    }

    public function listClients($parameters = null, $export = false)
    {
        $page = $parameters->has('page') ? $parameters->get('page') : 1;
        $limit = $parameters->has('limit') ? $parameters->get('limit') : config('app.paginate_by');

        $query = $this->obj
            ->select("sale_clients.*")
            ->where('sale_clients.deleted_at', null)
            ->where('sale_clients.id', '<>', config('app.client_default'));

        if($parameters != null){

            if (isset($parameters['search'])) {
                if($parameters['search'] != 'null'){
                    $text_search = $parameters->get('search');
                    if ($text_search === 'null') {
                        $text_search = null;
                    }
                    if ($text_search !== null) {
                        $text_search = strtoupper($text_search);

                        $query = $query->where(function ($sql) use($text_search) {
                            $sql->where('sale_clients.fullname', 'like', '%' . $text_search . '%')
                                ->orWhere('sale_clients.document', 'like', '%' . $text_search . '%');
                        });
                    }
                }
            }
        }

        $all_data = $query->get();

        if($export){
            $query = $query->orderBy('sale_clients.id', 'DESC')->get();
        }
        else{
            $query = $query
                ->skip(($page - 1) * $limit)
                ->take($limit)
                ->orderBy('sale_clients.id', 'DESC')->get()->toArray();
        }


        $response = array(
            'results' => $query,
            'count' => count($query),
            'total' => count($all_data),
            'paginate_by' => config('app.paginate_by')
        );

        return $response;
    }

    public function create(array $attributes)
    {
        $this->helperChecker->checkClientDocument($attributes['document']);
        $this->helperChecker->checkDocumentLengthByTypeDocument($attributes['document'], $attributes['type_document']);

        $item = new Client();
        $item->fullname = $attributes['fullname'];
        $item->document = $attributes['document'];
        $item->tradename = $attributes['tradename'];
        $item->typedocument_id = $attributes['type_document'];
        $item->telephone = $attributes['telephone'];
        $item->cellphone = $attributes['cellphone'];
        $item->email = $attributes['email'];
        $item->address = $attributes['address'];
        $item->type_client = $attributes['type_client'];
        $item->sex = $attributes['sex'];
        $item->status = $attributes['status'];
        $item->date_birthday = $attributes['date_birthday'];
        $item->url_image = null;
        $item->save();

        $this->saveUrlImage($item, $attributes);
        return $item;
    }

    public function createFromGuest(array $attributes)
    {
        $check_guest = $this->helperChecker->checkClientDocumentGuest($attributes['document']);
        $this->helperChecker->checkDocumentLengthByTypeDocument($attributes['document'], $attributes['typedocument_id']);

        if($check_guest["check"] === true){
            $item = new Client();
            $item->fullname = $attributes['fullname'];
            $item->document = $attributes['document'];
            $item->typedocument_id = $attributes['typedocument_id'];
            $item->cellphone = $attributes['cellphone'];
            $item->email = $attributes['email'];
            $item->sex = $attributes['sex'];
            $item->date_birthday = $attributes['date_birthday'];
            $item->save();
            return $item;
        }
        else{
           $item = $this->updateFromGuest($check_guest["data"][0]->id, $attributes);
           return $item;
        }
    }

    public function updateFromGuest($id, array $attributes)
    {
        $item = $this->obj->find($id);
        $item->fullname = $attributes['fullname'];
        $item->document = $attributes['document'];
        $item->typedocument_id = $attributes['typedocument_id'];
        $item->cellphone = $attributes['cellphone'];
        $item->email = $attributes['email'];
        $item->sex = $attributes['sex'];
        $item->date_birthday = $attributes['date_birthday'];
        $item->save();

        return $item;
    }

    public function updated($id, array $attributes)
    {
        $this->helperChecker->checkClientDocumentUpdate($attributes['document'], $id);
        $this->helperChecker->checkDocumentLengthByTypeDocument($attributes['document'], $attributes['type_document']);

        $item = $this->obj->find($id);
        $item->fullname = $attributes['fullname'];
        $item->document = $attributes['document'];
        $item->tradename = $attributes['tradename'];
        $item->typedocument_id = $attributes['type_document'];
        $item->telephone = $attributes['telephone'];
        $item->cellphone = $attributes['cellphone'];
        $item->email = $attributes['email'];
        $item->address = $attributes['address'];
        $item->type_client = $attributes['type_client'];
        $item->sex = $attributes['sex'];
        $item->status = $attributes['status'];
        $item->date_birthday = $attributes['date_birthday'];
        $item->url_image = null;
        $item->save();

        $this->saveUrlImage($item, $attributes);
        return $item;
    }

    public function updatedFromSale($id, array $attributes)
    {
        $item = $this->obj->find($id);
        $item->document = $attributes['document'];
        $item->address = $attributes['address'];
        $item->url_image = null;
        $item->save();
        return $item;
    }

    public function updatedFromProforma($id, array $attributes)
    {
        $item = $this->obj->find($id);
        $item->email = $attributes['email'];
        $item->address = $attributes['address'];
        $item->cellphone = $attributes['cellphone'];
        $item->save();
        return $item;
    }


    public function saveUrlImage($item, $attributes)
    {
        $image = $attributes['url_image'];
        if ($image != null) {
            $file_url = $this->storageHelper->saveUrlImgClient($attributes);
            $item->url_image = $file_url;
            $item->save();
        }
    }


    public function find($id)
    {
        $item = $this->obj->find($id);
        return $item;
    }

    public function deleted($id)
    {
        $item = $this->obj->find($id);
        $item->delete();
    }

    public function search($text)
    {
        $text_search = strtoupper($text);

        return $this->obj
            ->select('sale_clients.id as id', 'fullname', 'document', 'address', 'config_typedocument.id as typedocument_id','sale_clients.cellphone','sale_clients.email',
                DB::raw("concat(config_typedocument.description, ' : ' , sale_clients.document) AS str_document"))
            ->join('config_typedocument', 'sale_clients.typedocument_id', '=', 'config_typedocument.id')
            ->where('sale_clients.deleted_at', null)
            ->where('fullname', 'LIKE', '%' . $text_search . '%')
            ->orwhere('document', 'LIKE', '%' . $text_search . '%')
            ->get();
    }
}
